use core:lang;

/**
 * The range-based for-loop. Expands to (more or less):
 * var at = c.begin();
 * var end = c.end();
 * for (; at != end; ++i) { name = at.v; ... }
 *
 * Or, for range-style iterators:
 * var i = c.iter();
 * while (name = i.next) { ... }
 */
class RangeFor extends Breakable {
	init(SrcPos pos, Block parent, SStr name, Expr container) {
		init(pos, parent) {
			valName = name.v;
		}

		createCode(container);
	}

	init(SrcPos pos, Block parent, SStr key, SStr value, Expr container) {
		init(pos, parent) {
			keyName = key.v;
			valName = value.v;
		}

		createCode(container);
	}

	void body(Expr expr) {
		unless (loopBlock)
			return;

		loopBlock.add(expr);
	}

	// Hide the fact that we're actually two nested blocks!
	LocalVar? variable(SimplePart part) {
		if (part.name == valName) {
			if (loopBlock) {
				return loopBlock.variable(part);
			}
		}

		if (keyName) {
			if (part.name == keyName) {
				if (loopBlock) {
					return loopBlock.variable(part);
				}
			}
		}

		super:variable(part);
	}

	// Generate code.
	void blockCode(CodeGen state, CodeResult to) {
		// We put all initialization logic in its own child block, rather than one for each expression.
		CodeGen child = state.child();
		state.l << core:asm:begin(child.block);
		for (Nat i = 0; i < initExprs.count; i++) {
			initExprs[i].code(child, CodeResult());
		}
		state.l << core:asm:end(child.block);

		if (loop) {
			loop.code(state, to);
		}
	}

	// Support for break and continue.
	void willBreak() {
		if (loop)
			loop.willBreak();
	}
	void willContinue() {
		if (loop)
			loop.willContinue();
	}

	To breakTo() {
		if (loop)
			loop.breakTo();
		else
			throw InternalError("The loop was not initialized properly");
	}
	To continueTo() {
		if (loop)
			loop.continueTo();
		else
			throw InternalError("The loop was not initialized properly");
	}

private:

	// Names of the key and value variables.
	Str? keyName;
	Str valName;

	// Block containing the loop body.
	ExprBlock? loopBlock;

	// Expressions to execute in this block (we're basically an ExprBlock).
	Expr[] initExprs;

	// The loop itself. This is always the last expression in this block.
	Breakable? loop;


	void createCode(Expr container) {
		Var cont(this, SStr(" c", pos), container);
		LocalVarAccess getCont(pos, cont.var);
		initExprs.push(cont);

		if (createIterCode(getCont))
			return;

		if (createRangeCode(getCont))
			return;

		// Note: May not use fancy strings, their implementation depend on us!
		throw SyntaxError(pos, container.toS + " is not iterable.");
	}

	Bool createIterCode(LocalVarAccess container) {
		Expr beginExpr = namedExpr(this, SStr("begin", pos), container, Actuals());
		Expr endExpr = namedExpr(this, SStr("end", pos), container, Actuals());

		if (beginExpr as UnresolvedName)
			return false;
		if (endExpr as UnresolvedName)
			return false;

		Var atDecl(this, SStr(" at", pos), beginExpr);
		LocalVarAccess getAt(pos, atDecl.var);
		initExprs.push(atDecl);

		Var endDecl(this, SStr(" end", pos), endExpr);
		LocalVarAccess getEnd(pos, endDecl.var);
		initExprs.push(endDecl);

		For loop(pos, this);
		this.loop = loop;

		ExprBlock body(pos, loop);
		loop.test(namedExpr(loop, SStr("!=", pos), getAt, Actuals(getEnd)));
		loop.update(namedExpr(loop, SStr("++*", pos), getAt, Actuals()));
		loop.body(body);

		if (keyName) {
			body.add(Var(body, SStr(keyName, pos), namedExpr(loop, SStr("k", pos), getAt, Actuals())));
		}

		body.add(Var(body, SStr(valName), namedExpr(loop, SStr("v", pos), getAt, Actuals())));
		loopBlock = body;

		true;
	}

	Bool createRangeCode(LocalVarAccess container) {
		Expr iterExpr = namedExpr(this, SStr("iter", pos), container, Actuals());
		if (iterExpr as UnresolvedName)
			return false;

		Var iterDecl(this, SStr(" iter", pos), iterExpr);
		LocalVarAccess getIter(pos, iterDecl.var);

		Expr nextExpr = namedExpr(this, SStr("next", pos), getIter, Actuals());
		if (nextExpr as UnresolvedName)
			return false;

		initExprs.push(iterDecl);

		Loop loop(pos, this);
		this.loop = loop;

		if (keyName)
			throw SyntaxError(pos, "Keys are not supported for range-based iterators.");

		WeakMaybeCast cond(nextExpr);
		cond.name(SStr(valName, pos));
		loop.cond(cond);

		CondSuccess body(pos, loop, cond);
		loop.whileBody(body);

		ExprBlock eBody(pos, body);
		body.set(eBody);
		loopBlock = eBody;

		true;
	}

}
