use ui;
use core:geometry;
use graphics;

/**
 * A presentation.
 *
 * A presentation consists of a number of Slides, each of which contains a number of Elements to be
 * laid out in a certain way and displayed on the screen. Each slide may contain animations that
 * alter the state or layout of elements on the screen.
 *
 * In addition to the slides, the Presentation object contains global settings and resources for the
 * presentation in its entirety, such as background color, page numbering etc.
 */
class Presentation on Render {
	// Title of the presentation.
	Str title;

	// Size of the slides.
	Size size;

	// Border around slides.
	Size border;

	// Default text styles.
	TextStyle headingStyle;
	TextStyle contentStyle;

	// Default animation time.
	Duration animationTime;

	// All slides in the presentation.
	private Slide[] slides;

	// The background.
	private Slide myBackground;

	// Create.
	init(Str title) {
		Size size(640, 480);
		Size border(20, 20);

		layout:Border bgLayout;
		bgLayout.border = Size(0);
		bgLayout.add(component(SolidFill(white)));

		init() {
			title = title;
			size = size;
			border = border;
			myBackground = Slide(bgLayout);
			headingStyle = defaultHeadingStyle;
			contentStyle = defaultContentStyle;
			animationTime = 200 ms;
		}

		myBackground.added(this);
	}

	// Make the presentation into 16:9 format.
	void makeWide() {
		size = Size(size.h * 16 / 9, size.h);
	}

	// Make the presentation into 4:3 format.
	void makeNarrow() {
		size = Size(size.h * 4 / 3, size.h);
	}

	// Perform any cleanup necessary.
	void cleanup() {
		for (slide in slides)
			slide.cleanup();
	}

	// Attach a repaint callback if any elements require the ability.
	void setupRepaint(Fn<void> callback) {
		for (slide in slides)
			slide.setupRepaint(callback);
	}


	/**
	 * A cursor to a position inside a presentation, excluding the timecode of any current
	 * animation.
	 *
	 * The entire state is stored in this class, so that a single application may have multiple
	 * cursors into the same presentation. It is then possible to render the presentation at the
	 * different stages at will.
	 *
	 * Do not modify this object directly, use the corresponding functions inside the Presentation
	 * class instead.
	 */
	public class Cursor {
		// Current slide.
		package Nat slide;

		// Current animation inside the slide.
		package Nat step;

		// Create a state representing the first slide in a presentation.
		init() {
			init() {
				slide = 0;
				step = 0;
			}
		}

		// Convenience constructor for use internally.
		package init(Nat slide, Nat step) {
			init() {
				slide = slide;
				step = step;
			}
		}

		// To string.
		void toS(StrBuf to) {
			to << "{ slide " << slide << ", step " << step << " }";
		}

		// Compare.
		Bool ==(Cursor other) {
			slide == other.slide & step == other.step;
		}
	}

	// The background slide.
	Slide background() { myBackground; }

	// Set the background slide.
	assign background(Slide s) {
		myBackground = s;
		s.added(this);
	}

	// Add a slide.
	void add(Slide s) {
		slides << s;
		s.added(this);
	}

	// Draw the presentation in its current state. Assumes 'g' has been set up to match the slide
	// size. 'time' is a number in the interval [0, 1] that indicates the position of the current
	// animation. 1.0 means complete, and 0.0 means it has just started. 'pos' is the state of the
	// current slide, and 'last' is the state of the previous state in the presentation. This is
	// used to provide good slide transitions.
	void draw(Graphics g, Cursor pos, Cursor? last, Float time) {
		Nat slide = pos.slide;
		if (slide >= slides.count) {
			return;
		}

		Slide curr = slides[slide];
		if (pos.step == 0 & time < 1.0) {
			if (intro = curr.intro) {
				Duration d = max(curr.duration(0), intro.duration);
				intro.draw(this, g, pos, last, d * time);
				return;
			}
		}

		drawSlide(g, pos, curr.duration(pos.step) * time);
	}

	// Draw a slide, complete with background and any overlays. Expected to be used from intro
	// animations.
	void drawSlide(Graphics g, Cursor cursor, Duration time) {
		var slide = cursor.slide;
		if (slide < slides.count) {
			background.draw(g, slide, background.duration(slide));
			slides[cursor.slide].draw(g, cursor.step, time);
		}
	}

	// Draw a slide that is not the current one.
	void drawSlide(Graphics g, Cursor cursor) {
		if (cursor.slide < slides.count) {
			Slide s = slides[cursor.slide];
			drawSlide(g, cursor, s.duration(cursor.step));
		}
	}

	// Get the animation state representing the beginning of the presentation. Expected to be used
	// from intro animations.
	Cursor begin() {
		Cursor(0, 0);
	}

	// Get the state representing the end of the presentation.
	Cursor end() {
		Cursor(slides.count, 0);
	}

	// Get the state for the n:th slide.
	Cursor at(Nat id) {
		Cursor(min(id, slides.count), 0);
	}

	// Get the next state.
	Cursor next(Cursor s) {
		if (s.slide >= slides.count)
			return s;

		Slide slide = slides[s.slide];
		if (s.step + 1 >= slide.stepCount)
			Cursor(s.slide + 1, 0);
		else
			Cursor(s.slide, s.step + 1);
	}

	// Get the state representing a jump to the next slide, skipping any animations in the current slide.
	Cursor nextSlide(Cursor s) {
		Cursor(min(s.slide + 1, slides.count), 0);
	}

	// Get the previous state.
	Cursor prev(Cursor s) {
		// TODO: Do we want to go back one step, or the entire slide?
		if (s.step > 0) {
			Cursor(s.slide, 0);
		} else if (s.slide > 0) {
			Cursor(s.slide - 1, 0);
		} else {
			Cursor(0, 0);
		}
	}

	// Get the duration of a specific animation. Returns 0 ms if no animation is used for the
	// specified state.
	Duration duration(Cursor s) {
		if (s.slide >= slides.count)
			return Duration();

		Slide slide = slides[s.slide];
		Duration r = slide.duration(s.step);

		// Intro?
		if (s.step == 0)
			if (intro = slide.intro)
				r = max(r, intro.duration);

		r;
	}

	// Get an element from the current slide at a particular position. Does not examine the
	// background slide.
	Element? elementAt(Cursor pos, Point pt) {
		Nat slide = pos.slide;
		if (slide >= slides.count)
			return null;

		return slides[slide].elementAt(pt);
	}
}


/**
 * A description of text-styles. Used to specify defaults in a presentation so that one doesn't need
 * to specify the same style all over the place.
 */
class TextStyle on Render {
	// Font.
	Font font;

	// Brush used to fill the text.
	Brush fill;

	// Spacing.
	Float space;

	// Create.
	init(Font font, Brush fill, Float space) {
		init() {
			font = font;
			fill = fill;
			space = space;
		}
	}

	// Create.
	init(Font font, Color fill, Float space) {
		init() {
			font = font;
			fill = SolidBrush(fill);
			space = space;
		}
	}
}

// Default text styles.
TextStyle defaultHeadingStyle() on Render { TextStyle(Font("Arial", 32), black, 30); }
TextStyle defaultContentStyle() on Render { TextStyle(Font("Arial", 22), black, 10); }
