## TFSA-2022-002: Heap OOB read in shape inference for `ReverseSequence`

### CVE Number
CVE-2022-21728

### Impact
The [implementation of shape inference for `ReverseSequence`](https://github.com/tensorflow/tensorflow/blob/5100e359aef5c8021f2e71c7b986420b85ce7b3d/tensorflow/core/ops/array_ops.cc#L1636-L1671) does not fully validate the value of `batch_dim` and can result in a heap OOB read:

```python
import tensorflow as tf

@tf.function
def test():
  y = tf.raw_ops.ReverseSequence(
    input = ['aaa','bbb'],
    seq_lengths = [1,1,1],
    seq_dim = -10,
    batch_dim = -10 )
  return y

test()
```

There is a check to make sure the value of `batch_dim` does not go over the rank of the input, but there is no check for negative values:

```cc
  const int32_t input_rank = c->Rank(input);
  if (batch_dim >= input_rank) {
    return errors::InvalidArgument(
        "batch_dim must be < input rank: ", batch_dim, " vs. ", input_rank);
  }
  // ...

  DimensionHandle batch_dim_dim = c->Dim(input, batch_dim);
```

Negative dimensions are allowed in some cases to mimic Python's negative indexing (i.e., indexing from the end of the array), however if the value is too negative then [the implementation of `Dim`](https://github.com/tensorflow/tensorflow/blob/5100e359aef5c8021f2e71c7b986420b85ce7b3d/tensorflow/core/framework/shape_inference.h#L415-L428) would access elements before the start of an array:

```cc
  DimensionHandle Dim(ShapeHandle s, int64_t idx) {
    if (!s.Handle() || s->rank_ == kUnknownRank) {
      return UnknownDim();
    }
    return DimKnownRank(s, idx);
  }
 
  static DimensionHandle DimKnownRank(ShapeHandle s, int64_t idx) {
    CHECK_NE(s->rank_, kUnknownRank);
    if (idx < 0) {
      return s->dims_[s->dims_.size() + idx];
    }
    return s->dims_[idx];
  }
```

### Patches
We have patched the issue in GitHub commit [37c01fb5e25c3d80213060460196406c43d31995](https://github.com/tensorflow/tensorflow/commit/37c01fb5e25c3d80213060460196406c43d31995).

The fix will be included in TensorFlow 2.8.0. We will also cherrypick this commit on TensorFlow 2.7.1, TensorFlow 2.6.3, and TensorFlow 2.5.3, as these are also affected and still in supported range.

### For more information
Please consult [our security guide](https://github.com/tensorflow/tensorflow/blob/master/SECURITY.md) for more information regarding the security model and how to contact us with issues and questions.

### Attribution
This vulnerability has been reported by Yu Tian of Qihoo 360 AIVul Team.
