/*
 * Copyright (C) 2000-2025 the xine project
 *
 * This file is part of xine, a unix video player.
 *
 * xine is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * xine is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110, USA
 *
 */

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <stdarg.h>
#include <unistd.h>
#include <pthread.h>

#include "lang.h"

#include "common.h"
#include "dlist.h"
#include "osd.h"
#include "panel.h"
#include "videowin.h"
#include "actions.h"

#include "oxine/odk.h"
#define OSD_ALGO_PALETTE


#define OSD_BAR_WIDTH 336
#define OSD_BAR_HEIGHT 28
#define OSD_SINFO_LINES 5

#define OSD_MIN_WIN_WIDTH 300
#define OSD_MIN_FONT_SIZE  16
#define OSD_MAX_FONT_SIZE  64

#ifdef HAVE_PTR_TO_PROTECTED_VISIBILITY_FUNC
#  define _xine_osd_show_unscaled xine_osd_show_unscaled
#  define _xine_osd_show xine_osd_show
#  define _xine_get_audio_lang xine_get_audio_lang
#  define _xine_get_spu_lang xine_get_spu_lang
#else
static void _xine_osd_show_unscaled (xine_osd_t *osd, int64_t vpts) {
  return xine_osd_show_unscaled (osd, vpts);
}
static void _xine_osd_show (xine_osd_t *osd, int64_t vpts) {
  return xine_osd_show (osd, vpts);
}
static int _xine_get_audio_lang (xine_stream_t *stream, int channel, char *buf) {
  return xine_get_audio_lang (stream, channel, buf);
}
static int _xine_get_spu_lang (xine_stream_t *stream, int channel, char *buf) {
  return xine_get_spu_lang (stream, channel, buf);
}
#endif

typedef enum {
  OSD_sinfo = 0,
  OSD_bar,
  OSD_btext,
  OSD_status,
  OSD_info,
  OSD_last
} osd_item_t;

typedef union {
  uint16_t xy[2];
  uint32_t w;
} osd_xy_t;

typedef struct {
  xitk_dnode_t  node;
  xine_osd_t   *osd;
  uint8_t       font_set, font_want; /** << pixels */
  uint8_t       visible; /** << 0 (hidden) or (seconds_remaining + 1) */
  uint8_t       doshow; /** << moved, resized, redrawn */
  osd_xy_t      xy;
  uint16_t      vis_w, vis_h;
} osd_object_t;

typedef enum {
  OSD_IDX_bg = 0,
  OSD_IDX_dark_gray,
  OSD_IDX_gray,
  OSD_IDX_yellow,
  OSD_IDX_white,
  OSD_IDX_pad,
  OSD_IDX_text,
  OSD_IDX_text_border,
  OSD_IDX_sharp,
  OSD_IDX_LAST
} osd_color_index_t;

struct xui_osd_s {
  pthread_mutex_t mutex;

  gGui_t         *gui;

  void          (*show) (xine_osd_t *osd, int64_t vpts);
  uint8_t         enabled;
  uint8_t         unscaled; /** << 2 (available) | 1 (user_enabled) */
  uint8_t         vwin_size_mode; /** << VWIN_SIZE_* */
  uint8_t         resizing;

  int             wsize[2];
  struct {
    uint8_t       side, top, bottom;
  }               margin;
  uint8_t         dummy;

  int             timeout;
  int             last_shown_time;

  xitk_dlist_t    pending;
  osd_object_t    obj[OSD_last];

  struct {
    char          buf[800];
    uint16_t      s[OSD_SINFO_LINES + 1], n;
  }               sinfo;

  struct {
    char          text[800];
    unsigned int  len;
  }               msg;

  struct {
    char          title[256];
    int           min, max, _val;
    osd_bar_type_t _type, type;
    unsigned int  val;
#define BAR_LINE_WIDTH 3
#define BAR_LINE_SPC 4
/* make sure to be odd. */
#define BAR_LINE_NUM ((((OSD_BAR_WIDTH - 2 * 3 + BAR_LINE_SPC) / (BAR_LINE_WIDTH + BAR_LINE_SPC) + 1) / 2 * 2) - 1)
#define BAR_LINE_X ((OSD_BAR_WIDTH - (BAR_LINE_WIDTH + BAR_LINE_SPC) * BAR_LINE_NUM + BAR_LINE_SPC) / 2)
    union { uint8_t b[BAR_LINE_NUM]; uint32_t w[(BAR_LINE_NUM + 3) / 4]; } ly1, ly2;
  }               bar;

  struct {
    char          buf[32];
    uint8_t       p, q;
  }               play_status;

#ifdef OSD_ALGO_PALETTE
  odk_palette_t   palette;
#else
  struct {
    uint8_t       index[OSD_IDX_LAST];
  }               palette;
#endif
};

static void _osd_set_pos (xui_osd_t *osd, osd_item_t item, int x, int y) {
  osd_xy_t xy = {{ x, y }};
  /* avoid unnecessary uv blending to yuv420 video. */
  xy.w &= 0xfffefffe;
  if (xy.w == osd->obj[item].xy.w)
    return;
  osd->obj[item].xy.w = xy.w;
  osd->obj[item].doshow = 1;
  xine_osd_set_position (osd->obj[item].osd, xy.xy[0], xy.xy[1]);
}

#ifndef OSD_ALGO_PALETTE
#  define CLUT_YCRCB(_y,_cr,_cb) {{ (_cb), (_cr), (_y), 0 }}
static const union {         /* CLUT == Color LookUp Table */
  uint8_t cb_cr_y_foo[4];
  uint32_t u32;
} text_color[ODK_PALETTE_SIZE] = {
  /* white, no border, translucid */
    CLUT_YCRCB (0x10, 0x80, 0x80), /** <<  0, background */
    CLUT_YCRCB (0x30, 0x80, 0x80), /** <<  1, dark gray shadow */
    CLUT_YCRCB (0x20, 0x80, 0x80), //2
    CLUT_YCRCB (0x80, 0x80, 0x80), //3
    CLUT_YCRCB (0x80, 0x80, 0x80), //4
    CLUT_YCRCB (0x80, 0x80, 0x80), //5
    CLUT_YCRCB (0x80, 0x80, 0x80), //6
    CLUT_YCRCB (0xa0, 0x80, 0x80), /** <<  7, gray line */
    CLUT_YCRCB (0xc0, 0x80, 0x80), //8
    CLUT_YCRCB (0xe0, 0x80, 0x80), /** <<  9, white line */
    CLUT_YCRCB (0xff, 0x80, 0x80), //10
  /* yellow, black border, transparent */
    CLUT_YCRCB (0x00, 0x00, 0x00), //0
    CLUT_YCRCB (0x80, 0x80, 0xe0), //1
    CLUT_YCRCB (0x80, 0x80, 0xc0), //2
    CLUT_YCRCB (0x60, 0x80, 0xa0), //3
    CLUT_YCRCB (0x40, 0x80, 0x80), //4
    CLUT_YCRCB (0x20, 0x80, 0x80), //5
    CLUT_YCRCB (0x00, 0x80, 0x80), //6
    CLUT_YCRCB (0x40, 0x84, 0x60), //7
    CLUT_YCRCB (0xd0, 0x88, 0x40), //8
    CLUT_YCRCB (0xe0, 0x8a, 0x00), //9
    CLUT_YCRCB (0xff, 0x90, 0x00), /** << 21, yellow line */
  /* white, blanck border, transparent, modified */
    CLUT_YCRCB ( 29, 131, 121), /** << 22  background */
    CLUT_YCRCB ( 15, 128, 128), // 1 d03
    CLUT_YCRCB ( 18, 128, 128), // 2 d05
    CLUT_YCRCB ( 23, 128, 128), // 3 d08
    CLUT_YCRCB ( 31, 128, 128), // 4 d19
    CLUT_YCRCB ( 59, 128, 128), // 5 d31
    CLUT_YCRCB ( 96, 128, 128), // 6 d55
    CLUT_YCRCB (151, 128, 128), // 7 d40
    CLUT_YCRCB (191, 128, 128), // 8 d18
    CLUT_YCRCB (219, 128, 128), // 9 d16
    CLUT_YCRCB (235, 128, 128), //10
};

#ifdef DEBUG
/* check text_color is properly packed */
typedef char _xitk_static_assertion_testpalettes_size[2*(!!(sizeof(text_color) == sizeof(uint32_t)*ODK_PALETTE_SIZE))-1];
#endif

static const uint8_t text_trans[ODK_PALETTE_SIZE] = {
  /* white, no border, translucid */
  0, 15, 5, 10, 15, 15, 15, 15, 15, 15, 15,
  /* yellow, black border, transparent */
  0, 8, 9, 10, 11, 12, 13, 14, 15, 15, 15,
  /* white, blanck border, transparent, modified */
  7, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15,
};
#endif

static void _osd_time2str (char **q, const int v[3], int speed) {
  char *p = *q;

  do {
    uint32_t u = (v[1] < 0) ? -v[1] : v[1];
    uint32_t r = (int32_t)u > v[2] ? u : (uint32_t)v[2];

    if (speed < XINE_FINE_SPEED_NORMAL * 4 / 5) {
      if (!u)
        break;
      /* .fff */
      *--p = u % 10u + '0';
      u /= 10u;
      *--p = u % 10 + '0';
      u /= 10u;
      *--p = u % 10u + '0';
      u /= 10u;
      *--p = '.';
    } else {
      u /= 1000u;
      if (!u)
        break;
    }
    /* m:ss[.fff] */
    *--p = u % 10u + '0';
    u /= 10u;
    *--p = u % 6u + '0';
    u /= 6u;
    *--p = ':';
    *--p = u % 10u + '0';
    u /= 10u;
    if (r < 10 * 60 * 1000)
      break;
    /* m[m:ss.fff] */
    *--p = u % 6u + '0';
    u /= 6u;
    if (r < 60 * 60 * 1000)
      break;
    /* h:[mm:ss.fff] */
    *--p = ':';
    *--p = u % 10u + '0';
    u /= 10u;
    if (r < 10 * 60 * 60 * 1000)
      break;
    /* h[h:mm:ss.fff] */
    *--p = u % 10u + '0';
    u /= 10u;
    if (r < 100 * 60 * 60 * 1000)
      break;
    /* h[hh:mm:ss.fff] */
    *--p = u + '0';
  } while (0);
  if (v[1] < 0)
    *--p = '-';
  *q = p;
}

#ifdef WITH_OSD_HIDE_ITEMS
static void _osd_obj_unset (osd_object_t *obj) {

  if (obj->node.next) {
    osd_object_t *o1 = (osd_object_t *)obj->node.next;
    if (o1->node.next)
      o1->visible += obj->visible - 1;
    xitk_dnode_remove (&obj->node);
    xine_osd_hide (obj->osd[0], 0);
    obj->visible = 0;
  }
}

void osd_hide_items (gGui_t *gui, uint32_t items) {
  if (!gui->osd)
    return;
  pthread_mutex_lock (&gui->osd->mutex);
  if (items & OSD_ITEM_SINFO)
    _osd_obj_unset (gui->osd->obj + OSD_sinfo);
  if (items & OSD_ITEM_BAR)
    _osd_obj_unset (gui->osd->obj + OSD_bar),
    _osd_obj_unset (gui->osd->obj + OSD_btext);
  if (items & OSD_ITEM_STATUS)
    _osd_obj_unset (gui->osd->obj + OSD_status);
  if (items & OSD_ITEM_INFO)
    _osd_obj_unset (gui->osd->obj + OSD_info);
  pthread_mutex_unlock (&gui->osd->mutex);
}
#endif

static void _osd_obj_set (xui_osd_t *osd, osd_item_t item, int timeout) {
  osd_object_t *obj = osd->obj + item, *o1 = (osd_object_t *)obj->node.next;

  if (o1) {
    if (o1->node.next)
      o1->visible += obj->visible - 1;
    xitk_dnode_remove (&obj->node);
    obj->visible = 0;
    if (obj->doshow)
      xine_osd_hide (obj->osd, 0);
  } else {
    obj->doshow = 1;
  }

  if (timeout > 0) {
    for (o1 = (osd_object_t *)osd->pending.head.next; o1->node.next; o1 = (osd_object_t *)o1->node.next) {
      int tt = timeout;
      timeout -= o1->visible - 1;
      if (timeout <= 0) {
        o1->visible -= tt;
        timeout = tt;
        break;
      }
    }
    o1 = (osd_object_t *)o1->node.prev;
    obj->visible = timeout + 1;
    xitk_dnode_insert_after (&o1->node, &obj->node);
    if (obj->doshow) {
      obj->doshow = 0;
      osd->show (obj->osd, 0);
    }
  }
}

static void _osd_enabled_cb (void *data, xine_cfg_entry_t *cfg) {
  xui_osd_t *osd = data;
  osd->enabled = cfg->num_value;
}

static int _osd_wsize_changed (xui_osd_t *osd, int lock) {
  int wsize[2], fonth1, fonth2;
  /* NOTE: most gui resize actions call osd_message () then _osd_wsize_changed (osd, 0).
   * later the x server echo calls osd_update () then _osd_wsize_changed (osd, 1),
   * which still needs to see whether we have a real resize or just a window move. */
  video_window_get_size (osd->gui->vwin, wsize, osd->vwin_size_mode);
  if (lock) {
    pthread_mutex_lock (&osd->mutex);
    if ((wsize[0] == osd->wsize[0]) && (wsize[1] == osd->wsize[1])) {
      pthread_mutex_unlock (&osd->mutex);
      return 0;
    }
    osd->wsize[0] = wsize[0];
    osd->wsize[1] = wsize[1];
  } else {
    if ((wsize[0] == osd->wsize[0]) && (wsize[1] == osd->wsize[1]))
      return 0;
    osd->wsize[0] = wsize[0] - 1;
    osd->wsize[1] = wsize[1] - 1;
  }
  osd->margin.side = wsize[0] * 30 / 1920;
  osd->margin.top  = wsize[1] * 20 / 1080;
  osd->margin.bottom = wsize[1] * 40 / 1080;

  fonth1 = 12 + (8 * wsize[1] + 360 * 4) / (720 * 4) * 4;
  fonth2 = 8 + (16 * wsize[1] + 360 * 4) / (720 * 4) * 4;
  if (fonth1 < OSD_MIN_FONT_SIZE)
    fonth1 = OSD_MIN_FONT_SIZE;
  else if (fonth1 > OSD_MAX_FONT_SIZE)
    fonth1 = OSD_MAX_FONT_SIZE;
  if (fonth2 < OSD_MIN_FONT_SIZE)
    fonth2 = OSD_MIN_FONT_SIZE;
  else if (fonth2 > OSD_MAX_FONT_SIZE)
    fonth2 = OSD_MAX_FONT_SIZE;

  if (osd->obj[OSD_info].font_want != fonth1)
    osd->obj[OSD_info].font_want = fonth1;
  if (osd->obj[OSD_sinfo].font_want != fonth1)
    osd->obj[OSD_sinfo].font_want = fonth1;
  if (osd->obj[OSD_status].font_want != fonth2)
    osd->obj[OSD_status].font_want = fonth2;
  if (fonth1 < 20)
    fonth1 = 20;
  if (osd->obj[OSD_btext].font_want != fonth1)
    osd->obj[OSD_btext].font_want = fonth1;

  if (osd->gui->verbosity >= 2)
    printf ("gui.osd.font.sizes (%d x %d) = %u, %u.\n", wsize[0], wsize[1],
      (unsigned int)osd->obj[OSD_info].font_want, (unsigned int)osd->obj[OSD_status].font_want);

  return 1;
}

static void _osd_set_mode (xui_osd_t *osd) {
  if (osd->unscaled == 3) {
    osd->vwin_size_mode = VWIN_SIZE_OUTPUT;
    osd->show = _xine_osd_show_unscaled;
  } else {
    osd->vwin_size_mode = VWIN_SIZE_FRAME;
    osd->show = _xine_osd_show;
  }
}

static void _osd_use_unscaled_cb (void *data, xine_cfg_entry_t *cfg) {
  gGui_t *gui = data;
  xui_osd_t *osd = gui->osd;
  int v;

  if (!osd)
    return;
  v = (osd->unscaled & ~1) | (cfg->num_value ? 1 : 0);
  if (v == osd->unscaled)
    return;

  pthread_mutex_lock (&osd->mutex);
  osd->unscaled = v;
  _osd_set_mode (osd);
  pthread_mutex_unlock (&osd->mutex);
}

static void _osd_timeout_cb (void *data, xine_cfg_entry_t *cfg) {
  xui_osd_t *osd = data;
  osd->timeout = cfg->num_value;
}

#ifdef OSD_ALGO_PALETTE
static void osd_set_colors (xui_osd_t *osd) {
  if (osd->obj[OSD_sinfo].osd)
    xine_osd_set_palette (osd->obj[OSD_sinfo].osd, osd->palette.yuv, osd->palette.o);
  if (osd->obj[OSD_bar].osd)
    xine_osd_set_palette (osd->obj[OSD_bar].osd, osd->palette.yuv, osd->palette.o);
  if (osd->obj[OSD_btext].osd)
    xine_osd_set_palette (osd->obj[OSD_btext].osd, osd->palette.yuv, osd->palette.o);
  if (osd->obj[OSD_info].osd)
    xine_osd_set_palette (osd->obj[OSD_info].osd, osd->palette.yuv, osd->palette.o);
  if (osd->obj[OSD_status].osd)
    xine_osd_set_palette (osd->obj[OSD_status].osd, osd->palette.yuv, osd->palette.o);
}

static const odk_user_color_t uc[] = {
  [OSD_IDX_sharp]       = {          0, "0text_sharpness",    N_("OSD text sharpness") },
  [OSD_IDX_bg]          = { 0x10808000, "1",                  NULL }, /* always transparent black. */
  [OSD_IDX_dark_gray]   = { 0x308080ff, "1color.dark_line",   N_("OSD info dark line color") },
  [OSD_IDX_gray]        = { 0xa08080ff, "1color.line",        N_("OSD info line color") },
  [OSD_IDX_white]       = { 0xeb8080ff, "1color.light_line",  N_("OSD info light line color") },
  [OSD_IDX_yellow]      = { 0xff9000ff, "1color.marked_line", N_("OSD info marked line color") },
  [OSD_IDX_pad]         = { 0x1d837977, "2color.pad",         N_("OSD info text pad color") },
  [OSD_IDX_text]        = { 0xeb8080ff, "3color.text",        N_("OSD info text color") },
  [OSD_IDX_text_border] = { 0x1d8379ff, "4color.text_border", N_("OSD info text outline color") }
};
#endif

static int _osd_rbox_h (int fontheight, int n) {
  int edge = (fontheight + 4) >> 3;
  return (2 * edge + n * fontheight + (n - 1) * (edge >> 1) + 1) & ~1;
}

void osd_init (gGui_t *gui) {
  xui_osd_t *osd = calloc (1, sizeof (*osd));

  if (!osd)
    return;

  osd->gui = gui;
  xitk_dlist_init (&osd->pending);
  if (xitk_init_NULL ()) {
    osd->obj[OSD_sinfo].node.next = osd->obj[OSD_sinfo].node.prev = NULL;
    osd->obj[OSD_bar].node.next = osd->obj[OSD_bar].node.prev = NULL;
    osd->obj[OSD_btext].node.next = osd->obj[OSD_btext].node.prev = NULL;
    osd->obj[OSD_status].node.next = osd->obj[OSD_status].node.prev = NULL;
    osd->obj[OSD_info].node.next = osd->obj[OSD_info].node.prev = NULL;
  }

  pthread_mutex_init (&osd->mutex, NULL);

#ifdef OSD_ALGO_PALETTE
  odk_palette_init (&osd->palette, gui->xine, "gui.osd.info.", uc, sizeof (uc) / sizeof (uc[0]));
#else
  /** bg 0, dark_gray 1, gray 7, yellow 21, white 9, pad 22, text 32, text_border 28, sharp 0 */
  memcpy (osd->palette.index, "\x00\x01\x07\x15\x09\x16\x20\x1c\0", OSD_IDX_LAST);
#endif

  osd->unscaled = xine_config_register_bool (gui->xine, "gui.osd_use_unscaled", 1,
    _("Use unscaled OSD"),
    _("Use unscaled (full screen resolution) OSD if possible"),
    CONFIG_LEVEL_ADV, _osd_use_unscaled_cb, gui) ? 1 : 0;

  /* not used for text. */
  osd->bar.type = OSD_BAR_LAST;
  osd->obj[OSD_bar].osd = xine_osd_new (gui->stream, 0, 0, OSD_BAR_WIDTH + 1, OSD_BAR_HEIGHT + 1);
  if (osd->obj[OSD_bar].osd) {
#ifndef OSD_ALGO_PALETTE
    xine_osd_set_palette (osd->obj[OSD_bar].osd, &text_color[0].u32, text_trans);
#endif
    osd->unscaled = (osd->unscaled & ~2)
                  | xitk_bitmove (xine_osd_get_capabilities (osd->obj[OSD_bar].osd), XINE_OSD_CAP_UNSCALED, 2);
  }
  _osd_set_mode (osd);

  osd->obj[OSD_sinfo].osd  = xine_osd_new (gui->stream, 0, 0, 1920, _osd_rbox_h (OSD_MAX_FONT_SIZE, OSD_SINFO_LINES));
  /* xine_osd_set_text_palette (osd->obj[OSD_sinfo].osd, XINE_TEXTPALETTE_WHITE_BLACK_TRANSPARENT, XINE_OSD_TEXT1); */
  osd->obj[OSD_btext].osd  = xine_osd_new (gui->stream, 0, 0,  960, _osd_rbox_h (OSD_MAX_FONT_SIZE, 1));
  osd->obj[OSD_status].osd = xine_osd_new (gui->stream, 0, 0,  480, _osd_rbox_h (OSD_MAX_FONT_SIZE, 1));
  osd->obj[OSD_info].osd   = xine_osd_new (gui->stream, 0, 0, 1920, _osd_rbox_h (OSD_MAX_FONT_SIZE, 1));
#ifndef OSD_ALGO_PALETTE
  if (osd->obj[OSD_info].osd)
    xine_osd_set_palette (osd->obj[OSD_info].osd, &text_color[0].u32, text_trans);
  if (osd->obj[OSD_status].osd)
    xine_osd_set_palette (osd->obj[OSD_status].osd, &text_color[0].u32, text_trans);
  if (osd->obj[OSD_btext].osd)
    xine_osd_set_palette (osd->obj[OSD_btext].osd, &text_color[0].u32, text_trans);
  if (osd->obj[OSD_sinfo].osd)
    xine_osd_set_palette (osd->obj[OSD_sinfo].osd, &text_color[0].u32, text_trans);
#else
  odk_palette_update (&osd->palette);
  osd_set_colors (osd);
#endif
  _osd_wsize_changed (osd, 0);

  osd->enabled = xine_config_register_bool (gui->xine, "gui.osd_enabled", 1,
    _("Enable OSD support"),
    _("Enabling OSD permit one to display some status/informations in output window."),
    CONFIG_LEVEL_BEG, _osd_enabled_cb, osd);

  osd->timeout = xine_config_register_num (gui->xine, "gui.osd_timeout", 3,
    _("Dismiss OSD time (s)"),
    _("Persistence time of OSD visual, in seconds."),
    CONFIG_LEVEL_BEG, _osd_timeout_cb, osd);

  /* NOTE: panel slider thread may use osd_tick () before this call. */
  gui->osd = osd;
}

void osd_hide (gGui_t *gui) {
  if (!gui->osd)
    return;
  pthread_mutex_lock (&gui->osd->mutex);
  while (1) {
    osd_object_t *obj = (osd_object_t *)gui->osd->pending.head.next;

    if (!obj->node.next)
      break;
    xitk_dnode_remove (&obj->node);
    xine_osd_hide (obj->osd, 0);
    obj->visible = 0;
  }
  pthread_mutex_unlock (&gui->osd->mutex);
}

void osd_deinit (gGui_t *gui) {
  unsigned int u;
  xui_osd_t *osd = gui->osd;

  if (!osd)
    return;
  osd_hide (gui);
  gui->osd = NULL;
#ifdef HAVE_XINE_CONFIG_UNREGISTER_CALLBACKS
  xine_config_unregister_callbacks (gui->xine, NULL, NULL, osd, sizeof (*osd));
#endif
  for (u = 0; u < OSD_last; u++)
    if (osd->obj[u].osd)
      xine_osd_free (osd->obj[u].osd);
  pthread_mutex_destroy (&osd->mutex);
  free (osd);
}

/** chain objects pending for timeout hide by relative time,
 *  + 1 to make sure obj.visible is always > 0. */
void osd_tick (gGui_t *gui) {
  xui_osd_t *osd = gui->osd;
  osd_object_t *obj;

  if (!osd)
    return;
  pthread_mutex_lock (&osd->mutex);
  do {
    obj = (osd_object_t *)osd->pending.head.next;
    if (&obj->node == &osd->pending.tail)
      break;
    if (--obj->visible > 1)
      break;
    do {
      xitk_dnode_remove (&obj->node);
      obj->visible = 0;
      xine_osd_hide (obj->osd, 0);
      obj = (osd_object_t *)osd->pending.head.next;
    } while ((&obj->node != &osd->pending.tail) && (obj->visible <= 1));
  } while (0);
#ifdef OSD_ALGO_PALETTE
  if ((&obj->node == &osd->pending.tail) && odk_palette_update (&osd->palette)) {
    osd_set_colors (osd);
    if (gui->verbosity >= 2)
      printf ("gui.osd.colors.chaged.\n");
  }
#endif
  pthread_mutex_unlock (&osd->mutex);
}

typedef struct {
  int x, y, w, h, n;
} osd_rect_t;

/** input:  x, y: rect pos, w, h: biggest line size, n: line count.
 *  output: x, y: first line pos, h: line stride. */
static void _osd_round_box (xui_osd_t *osd, osd_item_t type, osd_rect_t *r) {
  osd_object_t *obj = osd->obj + type;
  int edge = (r->h + 4) >> 3;
  r->h += edge >> 1;
  obj->vis_w = (r->w + 2 * edge + 1) & ~1;
  obj->vis_h = (r->h * r->n + 2 * edge - (edge >> 1) + 1) & ~1;
  xine_osd_draw_rect (obj->osd, r->x + 4, r->y,
    r->x + obj->vis_w - 4, r->y + 1, osd->palette.index[OSD_IDX_pad], 1);
  xine_osd_draw_rect (obj->osd, r->x + 2, r->y + 1,
    r->x + obj->vis_w - 2, r->y + 2, osd->palette.index[OSD_IDX_pad], 1);
  xine_osd_draw_rect (obj->osd, r->x + 1, r->y + 2,
    r->x + obj->vis_w - 1, r->y + 4, osd->palette.index[OSD_IDX_pad], 1);
  xine_osd_draw_rect (obj->osd, r->x, r->y + 4,
    r->x + obj->vis_w, r->y + obj->vis_h - 4, osd->palette.index[OSD_IDX_pad], 1);
  xine_osd_draw_rect (obj->osd, r->x + 1, r->y + obj->vis_h - 4,
    r->x + obj->vis_w - 1, r->y + obj->vis_h - 2, osd->palette.index[OSD_IDX_pad], 1);
  xine_osd_draw_rect (obj->osd, r->x + 2, r->y + obj->vis_h - 2,
    r->x + obj->vis_w - 2, r->y + obj->vis_h - 1, osd->palette.index[OSD_IDX_pad], 1);
  xine_osd_draw_rect (obj->osd, r->x + 4, r->y + obj->vis_h - 1,
    r->x + obj->vis_w - 4, r->y + obj->vis_h, osd->palette.index[OSD_IDX_pad], 1);
  r->x += edge;
  r->y += edge;
  obj->doshow = 1;
}

void osd_stream_infos (gGui_t *gui) {
  xui_osd_t *osd = gui->osd;
  osd_rect_t r = {.n = 0};
  int resizing;

  if (!osd)
    return;
  if (!osd->enabled || !osd->obj[OSD_sinfo].osd)
    return;

  resizing = osd->resizing;
  if (!resizing) {
    const char *input, *container, *vcodec, *acodec;
    char *p, *e;
    int v[3];
    int ints[8] = { XINE_STREAM_INFO_VIDEO_WIDTH,   XINE_STREAM_INFO_VIDEO_HEIGHT,
                    XINE_STREAM_INFO_VIDEO_BITRATE, XINE_STREAM_INFO_FRAME_DURATION,
                    XINE_STREAM_INFO_AUDIO_SAMPLERATE, XINE_STREAM_INFO_AUDIO_BITRATE,
                    XINE_STREAM_INFO_AUDIO_CHANNELS, -1};
#ifdef XINE_QUERY_STREAM_INFO
    char sbuf[512];
    int strings[5] = {XINE_META_INFO_INPUT_PLUGIN, XINE_META_INFO_SYSTEMLAYER,
      XINE_META_INFO_VIDEOCODEC, XINE_META_INFO_AUDIOCODEC, -1};
    xine_query_stream_info (gui->stream, sbuf, sizeof (sbuf), strings, ints);
    input     = strings[0] ? sbuf + strings[0] : NULL;
    container = strings[1] ? sbuf + strings[1] : NULL;
    vcodec    = strings[2] ? sbuf + strings[2] : NULL;
    acodec    = strings[3] ? sbuf + strings[3] : NULL;
#else
    input     = xine_get_meta_info (gui->stream, XINE_META_INFO_INPUT_PLUGIN);
    container = xine_get_meta_info (gui->stream, XINE_META_INFO_SYSTEMLAYER);
    vcodec    = xine_get_meta_info (gui->stream, XINE_META_INFO_VIDEOCODEC);
    acodec    = xine_get_meta_info (gui->stream, XINE_META_INFO_AUDIOCODEC);
    for (x = 0; ints[x] >= 0; x++)
      ints[x] = xine_get_stream_info (gui->stream, ints[x]);
#endif
    ints[2] = (ints[2] + 500) / 1000;
    ints[3] = (ints[3] > 90) ? (90000 * 100 + (ints[3] >> 1)) / ints[3] : 0;
    ints[5] = (ints[5] + 500) / 1000;

    if (!gui_xine_get_pos_length (gui, gui->stream, v))
      return;

    /* We're in visual animation mode */
    if (!(ints[0] | ints[1])) {
      if (gui->visual_anim.running) {
        if (gui->visual_anim.enabled == 1) {
          vcodec = _("post animation");
          video_window_get_size (gui->vwin, ints + 0, VWIN_SIZE_FRAME);
        } else if (gui->visual_anim.enabled == 2) {
          vcodec  = xine_get_meta_info (gui->visual_anim.stream, XINE_META_INFO_VIDEOCODEC);
          ints[0] = xine_get_stream_info (gui->visual_anim.stream, XINE_STREAM_INFO_VIDEO_WIDTH);
          ints[1] = xine_get_stream_info (gui->visual_anim.stream, XINE_STREAM_INFO_VIDEO_HEIGHT);
        }
      } else {
        vcodec = _("unknown");
        video_window_get_size (gui->vwin, ints + 0, VWIN_SIZE_FRAME);
      }
    }

    e = osd->sinfo.buf + sizeof (osd->sinfo.buf) - 4;
    p = osd->sinfo.buf;
    osd->sinfo.s[0] = 0;

    gui_playlist_lock (gui);
    osd->sinfo.s[r.n + 1] = /* osd->sinfo.s[r.n] + */ strlcpy (p, (gui->is_display_mrl) ? gui->mmk.mrl : gui->mmk.ident, e - p) + 1;
    gui_playlist_unlock (gui);
    r.n++;

    if (input && container) {
      p = osd->sinfo.buf + osd->sinfo.s[r.n];
      osd->sinfo.s[r.n + 1] = osd->sinfo.s[r.n] + snprintf (p, e - p, "%s, %s.", input, container) + 1;
      r.n++;
    }

    if (vcodec && ints[0] && ints[1]) {
      unsigned int a = (unsigned int)ints[3] / 100u, b = (unsigned int)ints[3] % 100u;
      p = osd->sinfo.buf + osd->sinfo.s[r.n];
      if (!b)
        osd->sinfo.s[r.n + 1] = osd->sinfo.s[r.n]
          + snprintf (p, e - p, "%s: %dx%d, %ufps, %dkbps.", vcodec, ints[0], ints[1], a, ints[2]) + 1;
      else
        osd->sinfo.s[r.n + 1] = osd->sinfo.s[r.n]
          + snprintf (p, e - p, "%s: %dx%d, %u.%02ufps, %dkbps.", vcodec, ints[0], ints[1], a, b, ints[2]) + 1;
      r.n++;
    }

    if (acodec && ints[4]) {
      p = osd->sinfo.buf + osd->sinfo.s[r.n];
      osd->sinfo.s[r.n + 1] = osd->sinfo.s[r.n] + snprintf (p, e - p, "%s: %dHz, %dch, %dkbps.", acodec, ints[4], ints[6], ints[5]) + 1;
      r.n++;
    }

    e = osd->sinfo.buf + sizeof (osd->sinfo.buf) - 2 * XINE_LANG_MAX - 32;
    p = osd->sinfo.buf + osd->sinfo.s[r.n];
    p += strlcpy (p, _("Audio: "), e - p);
    if (p > e)
      p = e;
    p += gui_lang_str (gui, p, 0);
    memcpy (p, ", ", 2); p += 2;
    e += XINE_LANG_MAX + 2;
    p += strlcpy (p, _("Subtitles: "), e - p);
    if (p > e)
      p = e;
    p += gui_lang_str (gui, p, 1);
    memcpy (p, ".", 2); p += 2; /* e += XINE_LANG_MAX + 2; */
    osd->sinfo.s[r.n + 1] = p - osd->sinfo.buf + 1;
    r.n++;
    osd->sinfo.n = r.n;

    _osd_wsize_changed (osd, 0);
    osd_stream_position (gui, v);
  } else {
    r.n = osd->sinfo.n;
  }

  if (osd->obj[OSD_sinfo].font_set != osd->obj[OSD_sinfo].font_want) {
    resizing *= 2;
    osd->obj[OSD_sinfo].font_set = osd->obj[OSD_sinfo].font_want;
    xine_osd_set_font (osd->obj[OSD_sinfo].osd, "sans", osd->obj[OSD_sinfo].font_want);
  }

  {
    int i;
    uint32_t save;
    char *spos, *p;
    /* limit name/mrl line */
    spos = p = osd->sinfo.buf + osd->sinfo.s[0];
    memcpy (&save, p, 4);
    while (1) {
      xine_osd_get_text_size (osd->obj[OSD_sinfo].osd, p, &r.w, &r.h);
      if (r.w <= (osd->wsize[0] - 2 * osd->margin.side))
        break;
      memcpy (spos, &save, 4);
      spos = p;
      memcpy (&save, p, 4);
      memcpy (p, "\x00...", 4);
      p++;
    }
    if ((osd->sinfo.s[0] != p - osd->sinfo.buf) || (resizing != 1)) {
      osd->sinfo.s[0] = p - osd->sinfo.buf;
      /* find largest line */
      for (i = 1; i < osd->sinfo.n; i++) {
        int w, h;
        xine_osd_get_text_size (osd->obj[OSD_sinfo].osd, osd->sinfo.buf + osd->sinfo.s[i], &w, &h);
        if (w > r.w)
          r.w = w;
        if (h > r.h)
          r.h = h;
      }
      xine_osd_clear (osd->obj[OSD_sinfo].osd);
      /* draw box */
      _osd_round_box (osd, OSD_sinfo, &r);
      /* draw text */
      for (i = 0; i < osd->sinfo.n; i++) {
        xine_osd_draw_text (osd->obj[OSD_sinfo].osd, r.x, r.y, osd->sinfo.buf + osd->sinfo.s[i], osd->palette.index[OSD_IDX_pad]);
        r.y += r.h;
      }
    }
    /* unlimit name/mrl again for possible resize */
    osd->sinfo.s[0] = 0;
    memcpy (spos, &save, 4);
  }

  {
    int x = osd->wsize[0] - osd->obj[OSD_sinfo].vis_w - osd->margin.side;
    if (x < 0)
      x = 0;
    _osd_set_pos (osd, OSD_sinfo, x, osd->margin.top);
    if (!resizing)
      pthread_mutex_lock (&osd->mutex);
    /* show this much text a bit longer. */
    _osd_obj_set (osd, OSD_sinfo, (osd->timeout * osd->sinfo.n) >> 1);
    if (!resizing)
      pthread_mutex_unlock (&osd->mutex);
  }
}

void osd_bar (gGui_t *gui, const char *title, int min, int max, int val, osd_bar_type_t type) {
  xui_osd_t *osd = gui->osd;
  uint8_t bar_color[BAR_LINE_NUM];
  unsigned int d1, d2;
  int resizing;

  if (!osd || (type >= OSD_BAR_LAST))
    return;
  resizing = osd->resizing;

  if (osd->enabled) {
    if (!osd->obj[OSD_bar].osd)
      return;
    if (!resizing) {
      osd->bar.min = min;
      osd->bar.max = max;
      osd->bar._val = val;
      osd->bar._type = type;
      strlcpy (osd->bar.title, title ? title : "", sizeof (osd->bar.title));
      _osd_wsize_changed (osd, 0);
    } else {
      min = osd->bar.min;
      max = osd->bar.max;
      val = osd->bar._val;
      type = osd->bar._type;
      title = osd->bar.title;
    }
    /* FIXME: still fixed size. */
    osd->obj[OSD_bar].vis_w = OSD_BAR_WIDTH;
    osd->obj[OSD_bar].vis_h = OSD_BAR_HEIGHT;
  } else if (resizing) {
    return;
  }

  if (min < max) {
    if (val < min)
      val = min;
    else if (val > max)
      val = max;
  } else {
    if (val < min)
      val = min = max - 1;
    else
      val = max = min + 1;
  }
  d2 = max - min;

  if (title) {
    static const int range[OSD_BAR_LAST][2] = {
      [OSD_BAR_PROGRESS]    = {    0, 100 },
      [OSD_BAR_POS2]        = { -100, 200 },
      [OSD_BAR_POS]         = {    0,   0 },
      [OSD_BAR_STEP_CENTER] = { -100, 200 },
      [OSD_BAR_STEPPER]     = {    0,  -1 }
    };
    char buf[1024];
    int num_min = min;
    unsigned int d3 = d2;
    osd_rect_t r = {.n = 1};

    if (range[type][1] >= 0)
      num_min = range[type][0], d3 = range[type][1];
    if (d3) {
      unsigned int tlen = xitk_find_0_or_byte (title, '%');
      if (!title[tlen]) {
        char *p = buf + sizeof (buf);

        val = ((unsigned int)(val - min) * d3 + (d2 >> 1)) / d2;
        val += num_min;
        d3 = (val < 0) ? -val : val;
        *--p = 0;
        do {
          *--p = d3 % 10u + '0';
          d3 /= 10u;
        } while (d3);
        if (val < 0)
          *--p = '-';
        *--p = ' ';
        *--p = ' ';
        if (tlen > (unsigned int)(p - buf))
          tlen = p - buf;
        p -= tlen;
        memcpy (p, title, tlen);
        title = p;
      }
    }
    if (!osd->enabled) {
      /* make sure buf stays valid through the next call
       * (compiler shall not optimize it to a stack frame leave plus jump). */
      panel_message (gui->panel, title);
      osd->dummy++;
      return;
    }
    if (osd->obj[OSD_btext].font_set != osd->obj[OSD_btext].font_want) {
      resizing *= 2;
      osd->obj[OSD_btext].font_set = osd->obj[OSD_btext].font_want;
      xine_osd_set_font (osd->obj[OSD_btext].osd, "sans", osd->obj[OSD_btext].font_want);
    }
    if (resizing != 1) {
      xine_osd_clear (osd->obj[OSD_btext].osd);
      xine_osd_get_text_size (osd->obj[OSD_btext].osd, title, &r.w, &r.h);
      _osd_round_box (osd, OSD_btext, &r);
      xine_osd_draw_text (osd->obj[OSD_btext].osd, r.x, r.y, title, osd->palette.index[OSD_IDX_pad]);
    }
    num_min = (osd->wsize[0] - osd->obj[OSD_btext].vis_w) >> 1;
    if (num_min < 0)
      num_min = 0;
    _osd_set_pos (osd, OSD_btext,
      num_min, osd->wsize[1] - osd->obj[OSD_bar].vis_h - osd->obj[OSD_btext].vis_h - osd->margin.bottom - 2);
  } else {
    osd->obj[OSD_btext].doshow = 0;
    if (!osd->enabled)
      return;
  }

  {
    static const uint8_t steps[OSD_BAR_LAST] = {
      [OSD_BAR_PROGRESS]    = BAR_LINE_NUM,
      [OSD_BAR_STEPPER]     = BAR_LINE_NUM,
      [OSD_BAR_STEP_CENTER] = BAR_LINE_NUM,
      [OSD_BAR_POS]         = BAR_LINE_NUM - 1,
      [OSD_BAR_POS2]        = BAR_LINE_NUM - 1
    };

    d1 = ((unsigned int)(val - min) * steps[type] + (d2 >> 1)) / d2;
    memset (bar_color, osd->palette.index[OSD_IDX_gray], sizeof (bar_color)); /* gray */
    if (steps[type] == BAR_LINE_NUM - 1) {
      /* pointer mode ----#-----. */
      bar_color[d1] = osd->palette.index[OSD_IDX_yellow]; /* yellow */
      /* yellow and white are hard to distinguish, so whiten the ends
       * only if pos is somewhere in between. */
      if (XITK_0_TO_MAX_MINUS_1 ((int)d1 - 1, BAR_LINE_NUM - 2))
        bar_color[0] = bar_color[BAR_LINE_NUM - 1] = osd->palette.index[OSD_IDX_white]; /* white */
    } else {
      /* bar mode ####-----. */
      if (d1 > 0)
        memset (bar_color, osd->palette.index[OSD_IDX_yellow], sizeof (bar_color[0]) * d1);
    }
  }

  if (!resizing)
    pthread_mutex_lock (&osd->mutex);
  if (type == osd->bar.type) {
    /* incremental */
    if (d1 != osd->bar.val) {
      unsigned int vold = osd->bar.val, vnew = d1;
      osd->obj[OSD_bar].doshow = 1;
      osd->bar.val = vnew;
      switch (type) {
        case OSD_BAR_POS2:
        default:
          osd->bar.ly1.b[vnew] = 2;
          osd->bar.ly2.b[vnew] = OSD_BAR_HEIGHT - 2;
          d1 = BAR_LINE_X + vold * (BAR_LINE_WIDTH + BAR_LINE_SPC);
          if ((vold != 0) && (vold != BAR_LINE_NUM - 1) && (vold != BAR_LINE_NUM / 2)) {
            osd->bar.ly1.b[vold] = 6;
            osd->bar.ly2.b[vold] = OSD_BAR_HEIGHT - 6;
            if (!resizing)
              pthread_mutex_unlock (&osd->mutex);
            xine_osd_draw_rect (osd->obj[OSD_bar].osd,
              d1, 2, d1 + BAR_LINE_WIDTH + 1, OSD_BAR_HEIGHT - 1, osd->palette.index[OSD_IDX_bg], 1);
            xine_osd_draw_rect (osd->obj[OSD_bar].osd,
              d1 + 1, 7, d1 + BAR_LINE_WIDTH + 1, OSD_BAR_HEIGHT - 5, osd->palette.index[OSD_IDX_dark_gray], 1);
          } else {
            if (!resizing)
              pthread_mutex_unlock (&osd->mutex);
          }
          if ((XITK_0_TO_MAX_MINUS_1 ((int)vold - 1, BAR_LINE_NUM - 2)) != (XITK_0_TO_MAX_MINUS_1 ((int)vnew - 1, BAR_LINE_NUM - 2))) {
            xine_osd_draw_rect (osd->obj[OSD_bar].osd,
              BAR_LINE_X, 2, BAR_LINE_X + BAR_LINE_WIDTH, OSD_BAR_HEIGHT - 2, bar_color[0], 1);
            xine_osd_draw_rect (osd->obj[OSD_bar].osd,
              BAR_LINE_X + (BAR_LINE_NUM - 1) * (BAR_LINE_WIDTH + BAR_LINE_SPC), 2,
              BAR_LINE_X + (BAR_LINE_NUM - 1) * (BAR_LINE_WIDTH + BAR_LINE_SPC) + BAR_LINE_WIDTH, OSD_BAR_HEIGHT - 2,
              bar_color[BAR_LINE_NUM - 1], 1);
          }
          xine_osd_draw_rect (osd->obj[OSD_bar].osd,
            d1, osd->bar.ly1.b[vold], d1 + BAR_LINE_WIDTH, osd->bar.ly2.b[vold], bar_color[vold], 1);
          d1 = BAR_LINE_X + vnew * (BAR_LINE_WIDTH + BAR_LINE_SPC);
          if ((vnew != 0) && (vnew != BAR_LINE_NUM - 1) && (vnew != BAR_LINE_NUM / 2)) {
            xine_osd_draw_rect (osd->obj[OSD_bar].osd,
              d1 + 1, osd->bar.ly1.b[vnew] + 1, d1 + BAR_LINE_WIDTH + 1, osd->bar.ly2.b[vnew] + 1,
              osd->palette.index[OSD_IDX_dark_gray], 1);
          }
          xine_osd_draw_rect (osd->obj[OSD_bar].osd,
            d1, osd->bar.ly1.b[vnew], d1 + BAR_LINE_WIDTH, osd->bar.ly2.b[vnew], bar_color[vnew], 1);
          break;
        case OSD_BAR_POS:
          if (!resizing)
            pthread_mutex_unlock (&osd->mutex);
          if ((XITK_0_TO_MAX_MINUS_1 ((int)vold - 1, BAR_LINE_NUM - 2)) != (XITK_0_TO_MAX_MINUS_1 ((int)vnew - 1, BAR_LINE_NUM - 2))) {
            xine_osd_draw_rect (osd->obj[OSD_bar].osd,
              BAR_LINE_X, 2, BAR_LINE_X + BAR_LINE_WIDTH, OSD_BAR_HEIGHT - 2, bar_color[0], 1);
            xine_osd_draw_rect (osd->obj[OSD_bar].osd,
              BAR_LINE_X + (BAR_LINE_NUM - 1) * (BAR_LINE_WIDTH + BAR_LINE_SPC), 2,
              BAR_LINE_X + (BAR_LINE_NUM - 1) * (BAR_LINE_WIDTH + BAR_LINE_SPC) + BAR_LINE_WIDTH, OSD_BAR_HEIGHT - 2,
              bar_color[BAR_LINE_NUM - 1], 1);
          }
          d1 = BAR_LINE_X + vold * (BAR_LINE_WIDTH + BAR_LINE_SPC);
          xine_osd_draw_rect (osd->obj[OSD_bar].osd,
            d1, osd->bar.ly1.b[vold], d1 + BAR_LINE_WIDTH, osd->bar.ly2.b[vold], bar_color[vold], 1);
          d1 = BAR_LINE_X + vnew * (BAR_LINE_WIDTH + BAR_LINE_SPC);
          xine_osd_draw_rect (osd->obj[OSD_bar].osd,
            d1, osd->bar.ly1.b[vnew], d1 + BAR_LINE_WIDTH, osd->bar.ly2.b[vnew], bar_color[vnew], 1);
          break;
        case OSD_BAR_PROGRESS:
          bar_color[0] = bar_color[BAR_LINE_NUM - 1] = osd->palette.index[OSD_IDX_white];
          /* fall through */
        case OSD_BAR_STEP_CENTER:
        case OSD_BAR_STEPPER:
          if (!resizing)
            pthread_mutex_unlock (&osd->mutex);
          if (vold > vnew)
            d1 = vold, vold = vnew, vnew = d1;
          for (d1 = BAR_LINE_X + vold * (BAR_LINE_WIDTH + BAR_LINE_SPC);
               vold < vnew; d1 += BAR_LINE_WIDTH + BAR_LINE_SPC, vold++)
            xine_osd_draw_rect (osd->obj[OSD_bar].osd,
              d1, osd->bar.ly1.b[vold], d1 + BAR_LINE_WIDTH, osd->bar.ly2.b[vold], bar_color[vold], 1);
          break;
      }
    } else {
      if (!resizing)
        pthread_mutex_unlock (&osd->mutex);
    }
  } else {
    /* full */
    osd->obj[OSD_bar].doshow = 1;
    osd->bar.type = type;
    osd->bar.val = d1;
    xine_osd_clear (osd->obj[OSD_bar].osd);
    switch (type) {
      case OSD_BAR_PROGRESS:
        bar_color[0] = bar_color[BAR_LINE_NUM - 1] = osd->palette.index[OSD_IDX_white];
        /* fall through */
      case OSD_BAR_POS:
        memset (osd->bar.ly1.w, 6, sizeof (osd->bar.ly1.w));
        osd->bar.ly1.b[0] = osd->bar.ly1.b[BAR_LINE_NUM - 1] = 2;
        memset (osd->bar.ly2.w, OSD_BAR_HEIGHT - 2, sizeof (osd->bar.ly2.w));
        if (!resizing)
          pthread_mutex_unlock (&osd->mutex);
        break;
      case OSD_BAR_POS2:
      default:
        memset (osd->bar.ly1.w, 6, sizeof (osd->bar.ly1.w));
        osd->bar.ly1.b[0] = osd->bar.ly1.b[BAR_LINE_NUM - 1] = osd->bar.ly1.b[BAR_LINE_NUM / 2] = osd->bar.ly1.b[d1] = 2;
        memset (osd->bar.ly2.w, OSD_BAR_HEIGHT - 6, sizeof (osd->bar.ly2.w));
        osd->bar.ly2.b[0] = osd->bar.ly2.b[BAR_LINE_NUM - 1] = osd->bar.ly2.b[BAR_LINE_NUM / 2] = osd->bar.ly2.b[d1] = OSD_BAR_HEIGHT - 2;
        if (!resizing)
          pthread_mutex_unlock (&osd->mutex);
        break;
      case OSD_BAR_STEP_CENTER:
        xine_osd_draw_rect (osd->obj[OSD_bar].osd,
          BAR_LINE_X + 1, 1, BAR_LINE_X + BAR_LINE_WIDTH + 1, OSD_BAR_HEIGHT - 4 + 1, osd->palette.index[OSD_IDX_dark_gray], 1);
        xine_osd_draw_rect (osd->obj[OSD_bar].osd,
          BAR_LINE_X + (BAR_LINE_WIDTH + BAR_LINE_SPC) * (BAR_LINE_NUM / 2) + 1, 1,
          BAR_LINE_X + (BAR_LINE_WIDTH + BAR_LINE_SPC) * (BAR_LINE_NUM / 2) + BAR_LINE_WIDTH + 1,
          OSD_BAR_HEIGHT - 4 + 1, osd->palette.index[OSD_IDX_dark_gray], 1);
        xine_osd_draw_rect (osd->obj[OSD_bar].osd,
          BAR_LINE_X, 0, BAR_LINE_X + BAR_LINE_WIDTH, OSD_BAR_HEIGHT - 4, osd->palette.index[OSD_IDX_gray], 1);
        xine_osd_draw_rect (osd->obj[OSD_bar].osd,
          BAR_LINE_X + (BAR_LINE_WIDTH + BAR_LINE_SPC) * (BAR_LINE_NUM / 2), 0,
          BAR_LINE_X + (BAR_LINE_WIDTH + BAR_LINE_SPC) * (BAR_LINE_NUM / 2) + BAR_LINE_WIDTH,
          OSD_BAR_HEIGHT - 4, osd->palette.index[OSD_IDX_gray], 1);
        /* fall through */
      case OSD_BAR_STEPPER:
        {
          /* even gcc 4.5 optimizes this to a set of build time constants.
           * the (admittedly more precise) byte wise version was left a runtime calculation loop. */
#define BAR_VMUL ((8.0 * (OSD_BAR_HEIGHT - 4)) / (BAR_LINE_NUM - 1))
          union { uint8_t b[4]; uint32_t w; } v = {{
            8 * (OSD_BAR_HEIGHT - 4) - (unsigned int)(0 * BAR_VMUL),
            8 * (OSD_BAR_HEIGHT - 4) - (unsigned int)(1 * BAR_VMUL),
            8 * (OSD_BAR_HEIGHT - 4) - (unsigned int)(2 * BAR_VMUL),
            8 * (OSD_BAR_HEIGHT - 4) - (unsigned int)(3 * BAR_VMUL)
          }};
          for (d1 = 0; d1 < (BAR_LINE_NUM + 3) / 4; v.w -= 0x01010101 * (unsigned int)(4 * BAR_VMUL), d1++)
            osd->bar.ly1.w[d1] = (v.w >> 3) & 0x1f1f1f1f;
        }
        memset (osd->bar.ly2.w, OSD_BAR_HEIGHT - 2, sizeof (osd->bar.ly2.w));
        if (!resizing)
          pthread_mutex_unlock (&osd->mutex);
        break;
    }
    {
      unsigned int i, x;
      for (x = BAR_LINE_X, i = 0; i < BAR_LINE_NUM; x += BAR_LINE_WIDTH + BAR_LINE_SPC, i++) {
        xine_osd_draw_rect (osd->obj[OSD_bar].osd,
          x + 1, osd->bar.ly1.b[i] + 1, x + BAR_LINE_WIDTH + 1, osd->bar.ly2.b[i] + 1, osd->palette.index[OSD_IDX_dark_gray], 1);
        xine_osd_draw_rect (osd->obj[OSD_bar].osd,
          x, osd->bar.ly1.b[i], x + BAR_LINE_WIDTH, osd->bar.ly2.b[i], bar_color[i], 1);
      }
    }
  }

  {
    int x = (osd->wsize[0] - osd->obj[OSD_bar].vis_w) >> 1;
    if (x < 0)
      x = 0;
    _osd_set_pos (osd, OSD_bar, x, osd->wsize[1] - osd->obj[OSD_bar].vis_h - osd->margin.bottom);

    /* don't even bother drawing osd over those small streams. it would look pretty bad. */
    if (osd->wsize[0] > OSD_MIN_WIN_WIDTH) {
      if (!resizing)
        pthread_mutex_lock (&osd->mutex);
      _osd_obj_set (osd, OSD_bar, osd->timeout);
      if (title && osd->obj[OSD_btext].osd)
        _osd_obj_set (osd, OSD_btext, osd->timeout);
      if (!resizing)
        pthread_mutex_unlock (&osd->mutex);
    }
  }
}

void osd_stream_position (gGui_t *gui, const int pos_time_length[3]) {
  char buf[256];
  const char *text;
  if (!gui->osd)
    return;
  osd_bar_type_t type;
  if (pos_time_length[1]) {
    char *p = buf + sizeof (buf);
    unsigned int t;
    p -= 2; memcpy (p, ")", 2);
    if (pos_time_length[2]) {
      unsigned int l;
      int u[3] = {0, pos_time_length[2], pos_time_length[2]};
      _osd_time2str (&p, u, XINE_FINE_SPEED_NORMAL);
      *--p = ' ';
      text = _("of");
      l = strlen (text);
      if (l > sizeof (buf) - 64)
        l = sizeof (buf) - 64;
      p -= l; memcpy (p, text, l);
      p -= 2; memcpy (p, "% ", 2);
      t = pos_time_length[1];
      l = pos_time_length[2];
      if (t < ~0u / 150u) {
        t *= 100u;
      } else {
        l /= 100u;
        l += !l;
      }
      t = (t + (l >> 1)) / l;
    } else {
      *--p = '%';
      t = pos_time_length[0];
      t = (t * 100u + 0x7fff) / 0xffff;
    }
    do {
      *--p = t % 10u + '0';
      t /= 10u;
    } while (t);
    p -= 2; memcpy (p, " (", 2);
    _osd_time2str (&p, pos_time_length, xine_get_param (gui->stream, XINE_PARAM_FINE_SPEED));
    text = p;
    type = OSD_BAR_POS2;
  } else {
    text = _("Position in Stream");
    type = OSD_BAR_POS;
  }
  osd_bar (gui, text, 0, 65535, pos_time_length[0], type);
}

void osd_message (gGui_t *gui, const char *info, ...) {
  xui_osd_t *osd = gui ? gui->osd : NULL;
  unsigned int l, resizing;

  if (!osd)
    return;
  if (gui->on_quit || !info || !osd->obj[OSD_info].osd)
    return;

  resizing = osd->resizing;
  if (!resizing) {
    char buf[sizeof (osd->msg.text)];
    const char *text;
    l = xitk_find_0_or_byte (info, '%');
    if (!info[l]) {
      text = info;
    } else {
      va_list args;
      va_start (args, info);
      if (!strcmp (info, "%s")) {
        text = va_arg (args, const char *);
        l = xitk_find_byte (text, 0);
      } else {
        l = vsnprintf (buf, sizeof (buf), info, args);
        text = buf;
      }
      va_end (args);
    }
    if (l > sizeof (osd->msg.text) - 1)
      l = sizeof (osd->msg.text) - 1;
    if ((osd->msg.len == l) && !memcmp (text, osd->msg.text, l)) {
      ;
    } else {
      osd->msg.len = l;
      memcpy (osd->msg.text, text, l);
      osd->msg.text[l] = 0;
      l = 0;
    }
    if (!osd->enabled)
      return panel_message (gui->panel, osd->msg.text);
    _osd_wsize_changed (osd, 0);
  } else {
    if (!osd->enabled)
      return;
    l = osd->msg.len;
  }

  if (osd->obj[OSD_info].font_set != osd->obj[OSD_info].font_want) {
    osd->obj[OSD_info].font_set = osd->obj[OSD_info].font_want;
    xine_osd_set_font (osd->obj[OSD_info].osd, "sans", osd->obj[OSD_info].font_want);
    l = 0;
  }
  if (!l) {
    osd_rect_t r = {.n = 1};
    xine_osd_clear (osd->obj[OSD_info].osd);
    xine_osd_get_text_size (osd->obj[OSD_info].osd, osd->msg.text, &r.w, &r.h);
    _osd_round_box (osd, OSD_info, &r);
    xine_osd_draw_text (osd->obj[OSD_info].osd, r.x, r.y, osd->msg.text, osd->palette.index[OSD_IDX_pad]);
  }

  if (!osd->obj[OSD_status].vis_h)
    osd->obj[OSD_status].vis_h = _osd_rbox_h (osd->obj[OSD_status].font_want, 1);
  _osd_set_pos (osd, OSD_info, osd->margin.side, osd->margin.top + osd->obj[OSD_status].vis_h + 3);

  if (!resizing)
    pthread_mutex_lock (&osd->mutex);
  _osd_obj_set (osd, OSD_info, osd->timeout);
  if (!resizing)
    pthread_mutex_unlock (&osd->mutex);
}

void osd_play_status (gGui_t *gui, int step) {
  xui_osd_t *osd = gui ? gui->osd : NULL;
  osd_rect_t r = {.n = 1};
  char *p, *q;
  int resizing;

  if (!osd)
    return;
  if (!osd->enabled || !osd->obj[OSD_status].osd)
    return;

  resizing = osd->resizing;
  if (!resizing) {
    int  status;

    p = osd->play_status.buf;
    status = xine_get_status (gui->stream);

    /*
      { : eject
      [ : previous
      | : | (thin)
      @ : | (large)
      ] : next
      } : stop
      $ : > (large)
      > : play
      < : pause
    */

    osd->play_status.buf[0] = 0;

    switch (status) {

      case XINE_STATUS_IDLE:
      case XINE_STATUS_STOP:
        memcpy (p, "}", 2);
        q = p + 1;
        break;

      case XINE_STATUS_QUIT:
        memcpy (p, "{", 2);
        q = p + 1;
        break;

      case XINE_STATUS_PLAY:
        {
          int speed = xine_get_param (gui->stream, XINE_PARAM_FINE_SPEED);
          int v[3] = {0, 0, 0};

          p = osd->play_status.buf + 32;
          *--p = 0;

          if (gui_xine_get_pos_length (gui, gui->stream, v)) {
            if (gui->verbosity >= 2)
              printf ("osd.status.update (now=%d, step=%d).\n", v[1], step);
            if (step) {
              if (step == 1) {
                v[1] += xine_get_stream_info (gui->stream, XINE_STREAM_INFO_FRAME_DURATION) / 90;
              } else if (step == 2) {
                if (v[1] <= osd->last_shown_time + 2) {
                  pthread_mutex_unlock (&osd->mutex);
                  return;
                }
              }
            }
            osd->last_shown_time = v[1];
            _osd_time2str (&p, v, speed);
            *--p = ' ';
          }
          q = p;

          if (speed < XINE_FINE_SPEED_NORMAL * 71 / 100) {
            if (speed <= 0) {
              p -= 1; memcpy (p, "<", 1); /* XINE_SPEED_PAUSE */
            } else if (speed < XINE_FINE_SPEED_NORMAL * 35 / 100) {
              p -= 3; memcpy (p, "@@>", 3); /* XINE_SPEED_SLOW_4 */
            } else {
              p -= 2; memcpy (p, "@>", 2); /* XINE_SPEED_SLOW_2 */
            }
          } else {
            if (speed < XINE_FINE_SPEED_NORMAL * 141 / 100) {
              p -= 1; memcpy (p, ">", 1); /* XINE_SPEED_NORMAL */
            } else if (speed < XINE_FINE_SPEED_NORMAL * 282 / 100) {
              p -= 2; memcpy (p, ">$", 2); /* XINE_SPEED_FAST_2 */
            } else {
              p -= 3; memcpy (p, ">$$", 3); /* XINE_SPEED_FAST_4 */
            }
          }
        }
        break;

      default: q = p;
    }
    osd->play_status.p = p - osd->play_status.buf;
    osd->play_status.q = q - osd->play_status.buf;
    _osd_wsize_changed (osd, 0);
    pthread_mutex_lock (&osd->mutex);
  } else {
    p = osd->play_status.buf + osd->play_status.p;
    q = osd->play_status.buf + osd->play_status.q;
  }

  if (osd->obj[OSD_status].font_set != osd->obj[OSD_status].font_want) {
    resizing *= 2; /** << resizing == 0 (new status), 1 (resizing but same text size), 2 (full). */
    osd->obj[OSD_status].font_set = osd->obj[OSD_status].font_want;
    xine_osd_set_font (osd->obj[OSD_status].osd, "cetus", osd->obj[OSD_status].font_want);
  }

  if (resizing != 1) {
    int h;
    xine_osd_clear (osd->obj[OSD_status].osd);
    /* set latin1 encoding (NULL) for status text with special characters,
     * then switch back to locale encoding ("") */
    xine_osd_set_encoding (osd->obj[OSD_status].osd, NULL);
#ifdef OSD_ALGO_PALETTE
    h = (odk_palette_get_index (&osd->palette, osd->palette.index[OSD_IDX_text_border]) >> 24)
      - (odk_palette_get_index (&osd->palette, osd->palette.index[OSD_IDX_text]) >> 24);
    if (h < 0)
      h = -h;
    if (*q && (h > 70)) {
      int w1, w2, h1, h2;
      /* use normal sans font for the time part, as it is a bit bolder, and looks better with outline. */
      *q = 0;
      xine_osd_get_text_size (osd->obj[OSD_status].osd, p, &w1, &h1);
      *q = ' ';
      xine_osd_set_font (osd->obj[OSD_status].osd, "sans", osd->obj[OSD_status].font_set);
      xine_osd_get_text_size (osd->obj[OSD_status].osd, q, &w2, &h2);
      r.w = w1 + w2;
      r.h = (h1 < h2) ? h2 : h1;
      h1 = (r.h - h1) >> 1;
      h2 = (r.h - h2) >> 1;
      _osd_round_box (osd, OSD_status, &r);
      xine_osd_draw_text (osd->obj[OSD_status].osd, r.x + w1, r.y + h2, q, osd->palette.index[OSD_IDX_pad]);
      *q = 0;
      xine_osd_set_font (osd->obj[OSD_status].osd, "cetus", osd->obj[OSD_status].font_set);
      xine_osd_draw_text (osd->obj[OSD_status].osd, r.x, r.y + h1, p, osd->palette.index[OSD_IDX_pad]);
      *q = ' ';
    } else
#endif
    {
      /* just the symbols (or no harsh outline). */
      xine_osd_get_text_size (osd->obj[OSD_status].osd, p, &r.w, &r.h);
      _osd_round_box (osd, OSD_status, &r);
      xine_osd_draw_text (osd->obj[OSD_status].osd, r.x, r.y, p, osd->palette.index[OSD_IDX_pad]);
    }
    xine_osd_set_encoding (osd->obj[OSD_status].osd, "");
  }

  _osd_set_pos (osd, OSD_status, osd->margin.side, osd->margin.top);
  /* don't even bother drawing osd over those small streams. it would look pretty bad. */
  if (osd->wsize[0] > OSD_MIN_WIN_WIDTH)
    _osd_obj_set (osd, OSD_status, osd->timeout);

  if (!resizing)
    pthread_mutex_unlock (&osd->mutex);
}

size_t gui_lang_str (gGui_t *gui, char buf[XINE_LANG_MAX], unsigned int audio0_spu1) {
  static const int xpar[2] = {XINE_PARAM_AUDIO_CHANNEL_LOGICAL, XINE_PARAM_SPU_CHANNEL};
  static int (* const xget[2]) (xine_stream_t *stream, int channel, char *buf) = {_xine_get_audio_lang, _xine_get_spu_lang};
  int channel = xine_get_param (gui->stream, xpar[audio0_spu1 &= 1]);

  if (channel != -2) {
    char t[16], *p;
    int r;
    unsigned int v;
    buf[0] = 0;
    r = xget[audio0_spu1] (gui->stream, channel, buf);
    if (r)
      return strlen (buf);
    if (channel == -1) {
      memcpy (buf, "auto", 5);
      return 4;
    }
    v = (channel < 0) ? -channel : channel;
    p = t + sizeof (t);
    *--p = 0;
    do {
      *--p = v % 10u + '0';
      v /= 10u;
    } while (v);
    if (channel < 0)
      *--p = '-';
    memcpy (buf, p, t + sizeof (t) - p);
    return t + sizeof (t) - p - 1;
  } else {
    memcpy (buf, "---", 4);
    return 3;
  }
}

void osd_lang (gGui_t *gui, unsigned int audio0_spu1) {
  static const char title[2][16] = { N_("Audio Channel: "), N_("Subtitles: ") };
  char buf[XINE_LANG_MAX];

  if (!gui->osd)
    return;
  audio0_spu1 &= 1;
  gui_lang_str (gui, buf, audio0_spu1);
  osd_message (gui, "%s%s", gettext (title[audio0_spu1]), get_language_from_iso639_1 (buf));
}

void osd_update (gGui_t *gui) {
  xui_osd_t *osd = gui ? gui->osd : NULL;

  if (!osd)
    return;
  if (!_osd_wsize_changed (osd, 1))
    return;
  if (osd->pending.head.next == &osd->pending.tail) {
    pthread_mutex_unlock (&osd->mutex);
    return;
  }

  osd->resizing = 1;

  if (osd->obj[OSD_status].visible)
    osd_play_status (gui, 0);
  else
    osd->obj[OSD_status].vis_h = 0;

  if (osd->obj[OSD_info].visible)
    osd_message (gui, "");

  if (osd->obj[OSD_sinfo].visible)
    osd_stream_infos (gui);

  if (osd->obj[OSD_bar].visible)
    osd_bar (gui, "", 0, 0, 0, 0);

  osd->resizing = 0;
  pthread_mutex_unlock (&osd->mutex);
}
